
# coding: utf-8

# # PyOpenCL: Experimenting in IPython

# In[ ]:

from __future__ import division
import numpy as np
import pyopencl as cl
import pyopencl.array


# Load the PyOpenCL IPython extension:

# In[2]:

get_ipython().magic('load_ext pyopencl.ipython_ext')


# Create an OpenCL context and a command queue:

# In[3]:

ctx = cl.create_some_context()
queue = cl.CommandQueue(ctx)


# ## Using the kernel 'magic'
# 
# Define an OpenCL kernel using the `%%cl_kernel` magic:

# In[4]:

get_ipython().run_cell_magic('cl_kernel', '', '\n__kernel void sum_vector(__global const float *a,\n__global const float *b, __global float *c)\n{\n  int gid = get_global_id(0);\n  c[gid] = a[gid] + b[gid];\n}')


# This looks for `cl_ctx` or `ctx` in the user namespace to find a PyOpenCL context.
# 
# Kernel names are automatically injected into the user namespace, so we can just use `sum_vector` from Python below.
# 
# Now create some data to work on:

# In[5]:

n = 10000

a = cl.array.empty(queue, n, dtype=np.float32)
a.fill(15)

b_host = np.random.randn(n).astype(np.float32)
b = cl.array.to_device(queue, b_host)

c = cl.array.empty_like(a)


# Run the kernel:

# In[6]:

sum_vector(queue, (n,), None, a.data, b.data, c.data)


# Check the result using `numpy` operations:

# In[7]:

assert (c.get() == b_host + 15).all()

